-- Copyright (c) 2010, Adam Crume
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
--     * Redistributions of source code must retain the above copyright notice,
--       this list of conditions and the following disclaimer.
--     * Redistributions in binary form must reproduce the above copyright
--       notice, this list of conditions and the following disclaimer in the
--       documentation and/or other materials provided with the distribution.
--     * Neither the name of the University of California nor the names of its
--       contributors may be used to endorse or promote products derived from
--       this software without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.


-- | This module supports linear math with any Fractional, not just Floats, so it can be used with things like Ratio Integer.
module LinearMath (
                   solveLinearSystem
                  ) where

import Data.List


-- | Solves a linear system.
solveLinearSystem :: (Fractional a) => [[a]] -> [a] -> [a]
solveLinearSystem [] [] = []
solveLinearSystem matrix vec =
    let (m, v) = reducedRowEchelon (matrix, vec)
    in take (length $ head m) v


-- | Moves the element at a specific index to the front.
moveToFront :: Int -> [a] -> [a]
moveToFront i xs =
    let (start, finish) = splitAt i xs
    in head finish : start ++ tail finish


-- | Moves the first non-zero row to the top.  If there is no non-zero row, and error results.
nonzeroFirstRow :: Num a => ([[a]], [a]) -> ([[a]], [a])
nonzeroFirstRow (matrix, vec) =
    case findIndex ((/= 0) . head) matrix of
      Just 0  -> (matrix, vec) -- Same as next line when i==0, but more efficient
      Just i  -> (moveToFront i matrix, moveToFront i vec)
      Nothing -> error "Under-constrained system"


-- | Puts the system in row-echelon form.  In other words, every leading coefficient is 1 and is strictly to the right of the leading coefficient of the row above.
--   Zero rows go to the bottom.
rowEchelon :: (Fractional a) => ([[a]], [a]) -> ([[a]], [a])
rowEchelon ([], []) = ([], [])
rowEchelon (matrix, vec)
    | null $ head matrix =
        if all (==0) vec
        then (matrix, vec)
        else error "Over-constrained system"
rowEchelon (matrix, vec) =
    let
        (matrix', vec') = nonzeroFirstRow (matrix, vec)
        firstrow = head matrix'
        firstc = head vec'
        firstrow' = map (/(head firstrow)) firstrow
        firstc' = firstc / (head firstrow)
        adjustRow (row, c) =
            let
                m = head row
                row' = eqZipWith (\x y -> x - m * y) row firstrow'
                c' = c - m * firstc'
            in
              (row', c')
        (bottommat, bottomvec) = unzip $ map adjustRow $ eqZip (tail matrix') (tail vec')
        bottomRight = (map tail bottommat, bottomvec)
        (bottommat', bottomvec') = rowEchelon bottomRight
    in
      (firstrow' : map (0:) bottommat', firstc' : bottomvec')


-- | Puts the system in reduced row-echelon form.  In other words, every leading coefficient is 1, it is the only non-zero coefficient in the row, and it is strictly to the right of the
--   coefficient in the row above.  Zero rows go to the bottom.
reducedRowEchelon :: (Fractional a) => ([[a]], [a]) -> ([[a]], [a])
reducedRowEchelon ([], []) = ([], [])
reducedRowEchelon (matrix, vec)
    | null $ head matrix =
        if all (==0) vec
        then (matrix, vec)
        else error "Over-constrained system"
reducedRowEchelon (matrix, vec) =
    let
        (mat', vec') = rowEchelon (matrix, vec)
        (tailmat, tailvec) = reducedRowEchelon (map tail $ tail mat', tail vec')
        firstrow = (head mat', head vec')
        separate (row', c') (row, c) =
            case findIndex (/= 0) row' of
              Just i -> (take i row ++ [0] ++ drop (i + 1) row, c - (row!!i) * c')
              Nothing -> (row, c)
        (firstrow', firstc') = foldr separate firstrow $ eqZip (map (0:) tailmat) tailvec
    in
      (firstrow' : map (0:) tailmat, firstc' : tailvec)


-- | Like zip, but gives an error when the list sizes do not match.
eqZip :: [a] -> [b] -> [(a, b)]
eqZip (x:xs) (y:ys) = (x, y):(eqZip xs ys)
eqZip [] [] = []
eqZip _ _ = error "List sizes do not match"


-- | Like zipWith, but gives an error when the list sizes do not match.
eqZipWith :: (a -> b -> c) -> [a] -> [b] -> [c]
eqZipWith f (x:xs) (y:ys) = (f x y):(eqZipWith f xs ys)
eqZipWith f [] [] = []
eqZipWith _ _ _ = error "List sizes do not match"
