-- Copyright (c) 2010, Adam Crume
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
--     * Redistributions of source code must retain the above copyright notice,
--       this list of conditions and the following disclaimer.
--     * Redistributions in binary form must reproduce the above copyright
--       notice, this list of conditions and the following disclaimer in the
--       documentation and/or other materials provided with the distribution.
--     * Neither the name of the University of California nor the names of its
--       contributors may be used to endorse or promote products derived from
--       this software without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.


module Ein.PatternHUnit where


import qualified Data.Map as Map
import Ein.Data
import Ein.Main
import Ein.Parse
import Ein.Pattern
import qualified Ein.StandardSymbols as S
import System.IO.Unsafe
import Test.HUnit


{-# NOINLINE context #-}
context = unsafePerformIO initialContextWithCore


testMatch pat exp matches =
    TestLabel (pat ++ ": " ++ exp) $
    TestCase $
    let
        ms = match (parse' pat) initialContext (parse' exp)
    in
      assertEqual ("Matching " ++ pat ++ " against " ++ exp ++ " was expected to yield " ++ show matches ++ ", but instead yielded " ++ show ms) matches ms


testRule rule exp value =
    TestLabel (rule ++ ": " ++ exp) $
    TestCase $
    let
        f = ruleFunc (expressionRule $ parse' rule)
        value' = (\x -> case x of Just (e, c) -> Just e; Nothing -> Nothing) $ f (parse' exp) emptyContext
        expected = case value of Nothing -> Nothing; Just y -> Just (parse' y)
    in
      assertEqual ("Applying " ++ rule ++ " to " ++ exp ++ " was expected to yield " ++ show expected ++ ", but instead yielded " ++ show value') expected value'


tests =
    let
        a = ESym "a"
        b = ESym "b"
        c = ESym "c"
        x = ESym "x"
        y = ESym "y"
        r = expressionRule $ parse' "x -> 1"
    in
      TestList [testMatch "1" "{1,a}" [],
                testMatch "1" "1" [Map.empty],
                testMatch "_" "1" [Map.empty],
                testMatch "x_" "1" [Map.singleton x [EInt 1]],
                testMatch "x_Integer|y_Symbol" "1" [Map.fromList [(x, [EInt 1]), (y, [])]],
                testMatch "x_Integer|y_Symbol" "z" [Map.fromList [(x, []), (y, [ESym "z"])]],
                testMatch "{x__,y__}" "{a,b,c}" [Map.fromList [(x, [a]), (y, [b, c])],
                                                 Map.fromList [(x, [a, b]), (y, [c])]],
                testMatch "{x__,y__,x__}" "{a,a,a,a,a}" [Map.fromList [(x, [a]), (y, [a, a, a])],
                                                         Map.fromList [(x, [a, a]), (y, [a])]],
                testMatch "{x__,y__,x__}" "{a,b,b,b,a}" [Map.fromList [(x, [a]), (y, [b, b, b])]],
                testMatch "{Pattern[x,Repeated[_,_]],__}" "{a,a,a,a,a}" [Map.fromList [(x, [a, a])],
                                                                         Map.fromList [(x, [a, a, a, a])]],
                testMatch "{Pattern[x,Repeated[_,_]],__}" "{a,a,a,a}" [Map.fromList [(x, [a, a])]],
                testMatch "{x___,y___}" "{a,b,c}" [Map.fromList [(x, []), (y, [a, b, c])],
                                                   Map.fromList [(x, [a]), (y, [b, c])],
                                                   Map.fromList [(x, [a, b]), (y, [c])],
                                                   Map.fromList [(x, [a, b, c]), (y, [])]],
                testMatch "x_/;x>0" "0" [],
                testMatch "x_/;x>0" "1" [Map.singleton x [EInt 1]],
                testRule "x->1" "x" (Just "1"),
                testRule "x->1" "y" Nothing,
                testRule "x->1" "x+1" Nothing] -- Rules aren't automatically recursive, so "x->1" doesn't apply to "x+1"
