-- Copyright (c) 2010, Adam Crume
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
--     * Redistributions of source code must retain the above copyright notice,
--       this list of conditions and the following disclaimer.
--     * Redistributions in binary form must reproduce the above copyright
--       notice, this list of conditions and the following disclaimer in the
--       documentation and/or other materials provided with the distribution.
--     * Neither the name of the University of California nor the names of its
--       contributors may be used to endorse or promote products derived from
--       this software without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.


module Ein.Pattern (
                    match,
                    expressionRule,
                    expressionRule',
                    genRep,
                    genRep',
                    variableNames
                   ) where

import qualified Data.Map as Map
import qualified Data.Set as Set
import Ein.Data
import {-# SOURCE #-} Ein.Main
import qualified Ein.StandardSymbols as S


-- | Matches a pattern against an expression.
match :: Exp -- ^ Pattern
      -> Context -- ^ Current context (needed for conditions)
      -> Exp -- ^ Expression to match against
      -> [Map.Map Exp [Exp]] -- ^ List of matches.  Each match is a map from variable names to the expressions they match (usually one unless it was a sequence pattern).
match pat c e =
    let
        ms = match' c [pat] [e] Map.empty []
        names = variableNames pat
        -- Makes sure that any variables that weren't matched are mapped to an empty list
        addDefault m name = Map.alter (\x -> case x of Nothing -> Just []; Just y -> Just y) name m
        fixMap m = foldl addDefault m (Set.toList names)
    in
      map fixMap ms


-- | Matches a sequence of patterns against a sequence of expressions.
match' :: Context -- ^ Current context
       -> [Exp] -- ^ Patterns
       -> [Exp] -- ^ Expressions to match against
       -> Map.Map Exp [Exp] -- ^ Map of variables that have already been matched to the expressions they matched
       -> [Exp] -- ^ Variables currently being matched
       -> [Map.Map Exp [Exp]] -- ^ List of matches.  Each match is a map from variable names to the expressions they match (usually one unless it was a sequence pattern).
match' _ [] [] vars names = [vars]
match' _ [] xs vars names = []
match' context ((EFunc (ESym head) args) : ps) exps vars names =
    case head of
      "Blank" -> case (args, exps) of
                   ((x:xs), (e:es)) | x /= expHead e -> []
                   (_, (e:es)) -> mergeMatch context ps es vars names [e] []
                   (_, []) -> []
      "Pattern" -> case args of
                     (name:pat:[]) -> match' context (pat:ps) exps vars (name:names)
                     _ -> []
      "Alternatives" -> concatMap (\arg -> match' context (arg:ps) exps vars names) args
      "Repeated" -> let subseqs = map (\x -> splitAt x exps) [(length args), 2 * (length args)..(length exps)]
                    in
                      concatMap (\(x,y) -> mergeMatch context ps y vars names x []) subseqs
      "RepeatedNull" -> let subseqs = map (\x -> splitAt x exps) [0, (length args)..(length exps)]
                        in
                          concatMap (\(x,y) -> mergeMatch context ps y vars names x []) subseqs
      "Condition" -> case args of
                       (pat:cond:[]) -> let m = match' context (pat:ps) exps vars names
                                            buildRule (var, vals) = expressionRule $ EFunc S.rule [var, squeeze vals]
                                            matchCondition vars = S.true == fst (eval' (genRep vars cond) context [])
                                        in
                                          filter matchCondition m
      otherwise -> match'' context (ESym head) args ps exps vars names
match' context ((EFunc head args) : ps) exps vars names = match'' context head args ps exps vars names
match' context (p:ps) [] vars names = []
match' context (p:ps) (e:es) vars names = if p == e then
                                              mergeMatch context ps es vars names [e] []
                                          else
                                              []


-- | Matches a plain function pattern and sequence of patterns against a sequence of expressions.
match'' :: Context -- ^ Current context
        -> Exp -- ^ Head of the pattern function
        -> [Exp] -- ^ Arguments of the pattern function
        -> [Exp] -- ^ Other patterns
        -> [Exp] -- ^ Expression to match against
        -> Map.Map Exp [Exp] -- ^ Map of variables that have already been matched to the expressions they matched
        -> [Exp] -- ^ Variables currently being matched
        -> [Map.Map Exp [Exp]] -- ^ List of matches.  Each match is a map from variable names to the expressions they match (usually one unless it was a sequence pattern).
match'' context head args ps (e@(EFunc eHead eArgs):es) vars names =
    let
        f ps es vars = mergeMatch context ps es vars names [e] []
        matchHead = f [head] [eHead] vars
        matchArgs = concatMap (f args eArgs) matchHead
        matchRest = concatMap (f ps es) matchArgs
    in
      matchRest
match'' context head args ps exps vars names = []


-- | Adds variable bindings and continues with the matching.  If a variable had an existing binding which is not the same as the new binding, matching fails.
-- patterns expressions existingVariables names newValue newNames
mergeMatch :: Context -- ^ Current context
           -> [Exp] -- ^ Patterns to match
           -> [Exp] -- ^ Expressions to match against
           -> Map.Map Exp [Exp] -- ^ Map of variables that have already been matched to the expressions they matched
           -> [Exp] -- ^ New variables to bind
           -> [Exp] -- ^ Value to bind for the new variables.  (Each variable gets bound to the whole list.)
           -> [Exp] -- ^ Variables currently being matched
           -> [Map.Map Exp [Exp]] -- ^ List of matches.  Each match is a map from variable names to the expressions they match (usually one unless it was a sequence pattern).
mergeMatch context ps es vars [] newValue newNames = match' context ps es vars newNames
mergeMatch context ps es vars names newValue newNames =
    let
        x = foldl f (Just vars) names
        f vars name = case vars of
                        Nothing -> Nothing
                        Just m -> case (Map.lookup name m) of
                                    Just oldValue | oldValue /= newValue -> Nothing
                                    otherwise -> Just (Map.insert name newValue m)
    in
      case x of
        Nothing -> []
        Just m -> match' context ps es m newNames


-- | Converts a Rule or RuleDelayed expression to a rule.
expressionRule :: Exp -> Rule
expressionRule e@(EFunc (ESym "Rule") (left:right:[])) = Rule (show e) $ expressionRule' left right
expressionRule e@(EFunc (ESym "RuleDelayed") (left:right:[])) = Rule (show e) $ expressionRule' left right


-- | Creates a rule function from a pattern expression and a replacement expression.
expressionRule' :: Exp -- ^ Pattern
                -> Exp -- ^ Replacement
                -> RuleFunction -- ^ Function that matches the pattern and replaces it with the replacement
expressionRule' left right =
    \x c -> 
    let matches = match left c x
    in case matches of
         (vars:others) -> case (genRep vars right) of
                            y | y == left -> Nothing
                            y -> Just (y, c)
         _ -> Nothing


-- | Replaces variables in an expression.  If the top level is a variable that matches zero expressions or multiple expressions, the result will be a Sequence.
genRep :: Map.Map Exp [Exp] -- ^ Map of variables to the expressions they matched
       -> Exp -- ^ Expression to replace variables in
       -> Exp -- ^ Expression with variables replaced
genRep vars template = squeeze $ genRep' vars [template]


-- | Replaces variables in expressions.  Note that the output list may not be the same length as the input list due to expressions matching empty sequences or multiple expressions.
genRep' :: Map.Map Exp [Exp] -- ^ Map of variables to the expressions they matched
        -> [Exp] -- ^ Expressions to replace variables in
        -> [Exp] -- ^ Expressions with variables replaced
genRep' vars template =
    let
        gen (EFunc head args) = let newHead = genRep vars head
                                    newArgs = genRep' vars args
                                in [EFunc newHead newArgs]
        gen x = case (Map.lookup x vars) of
                  Nothing -> [x]
                  Just y -> y
    in
      concatMap gen template


-- | Returns a set of all variable names in a pattern.
variableNames :: Exp -> Set.Set Exp
variableNames pat = f Set.empty pat where
    f set (EFunc (ESym "Pattern") (x:xs)) = foldl f (Set.insert x set) xs
    f set (EFunc head args) = foldl f set (head:args)
    f set _ = set
