-- Copyright (c) 2010, Adam Crume
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
--     * Redistributions of source code must retain the above copyright notice,
--       this list of conditions and the following disclaimer.
--     * Redistributions in binary form must reproduce the above copyright
--       notice, this list of conditions and the following disclaimer in the
--       documentation and/or other materials provided with the distribution.
--     * Neither the name of the University of California nor the names of its
--       contributors may be used to endorse or promote products derived from
--       this software without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.


module Ein.ParseHUnit where


import qualified Data.Map as Map
import Ein.Data
import Ein.Main
import Ein.Parse
import Ein.Pattern
import qualified Ein.StandardSymbols as S
import System.IO.Unsafe
import Test.HUnit
import Text.Parsec


{-# NOINLINE context #-}
context = unsafePerformIO initialContextWithCore


testParse s e =
    TestLabel s $
    TestCase $
    let
        e' = parse input "(unknown)" s
    in
      case e' of
        Left f -> assertFailure ("Error parsing " ++ s ++ ":\n" ++ (concatMap (\c -> case c of '\n' -> "\n\t"; x -> [x]) (show f)))
        Right x -> assertEqual ("Parsing " ++ s ++ " was expected to yield " ++ showExpanded e ++ ", but instead yielded " ++ showExpanded x) e x


tests = 
    let
        x = ESym "x"
        y = ESym "y"
        z = ESym "z"
    in
      TestList [testParse "(*comment*)x" x,
                testParse "-1" (EInt (-1)),
                testParse "x/;y" (EFunc (ESym "Condition") [x, y]),
                testParse "x||y" (EFunc (ESym "Or") [x, y]),
                testParse "x|y" (EFunc (ESym "Alternatives") [x, y]),
                testParse "x->y" (EFunc (ESym "Rule") [x, y]),
                testParse "x:>y" (EFunc (ESym "RuleDelayed") [x, y]),
                testParse "1+2*3" (EFunc (ESym "Plus") [EInt 1, EFunc (ESym "Times") [EInt 2, EInt 3]]),
                testParse "x_y" (EFunc (ESym "Pattern") [x, EFunc (ESym "Blank") [y]]),
                testParse "x__y" (EFunc (ESym "Pattern") [x, EFunc (ESym "Repeated") [EFunc (ESym "Blank") [y]]]),
                testParse "x___y" (EFunc (ESym "Pattern") [x, EFunc (ESym "RepeatedNull") [EFunc (ESym "Blank") [y]]]),
                testParse "_" (EFunc (ESym "Blank") []),
                testParse "__" (EFunc (ESym "Repeated") [EFunc (ESym "Blank") []]),
                testParse "___" (EFunc (ESym "RepeatedNull") [EFunc (ESym "Blank") []]),
                testParse "z[x, y[]]" (EFunc z [x, EFunc y []]),
                testParse "1+2/3" (EFunc (ESym "Plus") [EInt 1, EFunc (ESym "Times") [EInt 2, EFunc (ESym "Power") [EInt 3, EInt (-1)]]]),
                testParse "1-x" (EFunc (ESym "Plus") [EInt 1, EFunc (ESym "Times") [EInt (-1), x]]),
                testParse "z[x..,y...]" (EFunc z [EFunc (ESym "Repeated") [x], EFunc (ESym "RepeatedNull") [y]]),
                testParse "{x,y,(*comment*)z}" (EFunc (ESym "List") [x,y,z]),
                testParse "{x,y(*comment*),z}" (EFunc (ESym "List") [x,y,z]),
                testParse "x=y;x:=z" (EFunc (ESym "Compound") [EFunc (ESym "Set") [x,y], EFunc (ESym "SetDelayed") [x,z]]),
                testParse "x:=1" (EFunc (ESym "SetDelayed") [x, EInt 1]),
                testParse "x:=-1" (EFunc (ESym "SetDelayed") [x, EInt (-1)]),
                testParse "x=-1" (EFunc (ESym "Set") [x, EInt (-1)])]
