-- Copyright (c) 2010, Adam Crume
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
--     * Redistributions of source code must retain the above copyright notice,
--       this list of conditions and the following disclaimer.
--     * Redistributions in binary form must reproduce the above copyright
--       notice, this list of conditions and the following disclaimer in the
--       documentation and/or other materials provided with the distribution.
--     * Neither the name of the University of California nor the names of its
--       contributors may be used to endorse or promote products derived from
--       this software without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.


-- | Parser for the Ein language
module Ein.Parse (
                  input,
                  parse'
                 ) where

import Control.Applicative ((<$>))
import Control.Monad.Identity
import Text.Parsec hiding(string, token)
import qualified Text.Parsec.Char as C
import qualified Text.Parsec.Expr as PE
import qualified Text.Parsec.Language as L
import qualified Text.Parsec.Token as T
import Ein.Data
import qualified Ein.StandardSymbols as S


lexer :: T.TokenParser ()
lexer  = T.makeTokenParser (L.emptyDef {
                              T.commentStart = "(*",
                              T.commentEnd = "*)",
                              T.nestedComments = False,
                              T.identStart = letter,
                              T.identLetter = alphaNum,
                              T.reservedNames = [],
                              T.reservedOpNames = ["/;", "*","/","+","-"]
                            })

token = T.symbol lexer

reservedOp = T.reservedOp lexer

integer = do x <- T.integer lexer
             return (EInt x)

symbol = do x <- T.identifier lexer
            return (ESym x)

string = do x <- T.stringLiteral lexer
            return (EStr x)

reservedOp' s = try (C.string s >> T.whiteSpace lexer)

simpleExpression = PE.buildExpressionParser table functionCall
    where
         table = [ [ binop "^"  "Power"         PE.AssocLeft ],
                   [ PE.Prefix (reservedOp "-" >> return negate) ],
                   -- Since "/" has higher precedence and is a prefix of "/;", "/.", and "//.", and ";" is part of a non-operator rule,
                   -- we have to manually make sure we're not reading one of them
                   [ binop "*"  "Times"         PE.AssocLeft, PE.Infix (try (reservedOp "/" >> notFollowedBy (oneOf ";./") >> return divide)) PE.AssocLeft ],
                   [ binop "+"  "Plus"          PE.AssocLeft, binop' "-" minus PE.AssocLeft ],
                   [ binop "<"  "Less"          PE.AssocLeft, binop "<=" "LessOrEqual"       PE.AssocLeft,
                     binop ">"  "Greater"       PE.AssocLeft, binop ">=" "GreaterOrEqual"    PE.AssocLeft,
                     binop "==" "Equal"         PE.AssocLeft],
                   [ binop "&&" "And"           PE.AssocLeft ],
                   [ binop "||" "Or"            PE.AssocLeft ],
                   [ postop ".." "Repeated",                   postop "..." "RepeatedNull" ],
                   [ binop "|"  "Alternatives"  PE.AssocLeft ],
                   [ binop ":"  "Pattern"       PE.AssocNone ],
                   [ binop "/;" "Condition"     PE.AssocLeft ],
                   [ binop "->" "Rule"          PE.AssocRight, binop ":>"  "RuleDelayed"     PE.AssocRight ],
                   [ binop "/." "ReplaceAll"    PE.AssocLeft,  binop "//." "ReplaceRepeated" PE.AssocLeft ],
                   [ binopNF "=" "." "Set"      PE.AssocRight, binopNF ":=" "" "SetDelayed"  PE.AssocRight, postop "=." "Unset" ]
                 ]
         postop op head = PE.Postfix (reservedOp op >> return (\x -> EFunc (ESym head) [x]))
         binop' s f assoc = PE.Infix (reservedOp s >> return f) assoc
         binop s head assoc = binop' s (bin head) assoc
         -- Like binop, but reads s, not followed by any character in nf, followed by whitespace.
         binopNF s nf head assoc = PE.Infix (C.string s >> notFollowedBy (oneOf nf) >> T.whiteSpace lexer >> return (bin head)) assoc
         bin name x y = EFunc (ESym name) [x, y]
         bin2 name z x y = bin name x (z y)
         minus = bin2 "Plus" negate
         divide = bin2 "Times" (\y -> EFunc S.power [y, EInt (-1)])
         negate (EInt x) = EInt (-x)
         negate x = EFunc S.times [EInt (-1), x]


opChar = "~!@#$%^&*_+-=[]{}|./<>?"

-- Taken from http://www.mega-nerd.com/erikd/Blog/CodeHacking/Haskell/parsec_expression_parsing.html (with permission)
reservedOpNf :: String -> ParsecT String () Identity ()
reservedOpNf name = try (C.string name >> notFollowedBy (oneOf opChar) >> T.whiteSpace lexer)


primaryExpression =
                    try blankNullSequence <|>
                    try blankSequence <|>
                    blank <|>
                    between (token "(") (token ")") expression <|>
                    listExpression <|>
                    integer <|>
                    string <|>
                    do
                      x <- symbol
                      x2 <- option x (do
                                       y <- try blankNullSequence <|> try blankSequence <|> blank
                                       return $ EFunc (ESym "Pattern") [x, y]
                                       )
                      return x2

--input :: Parser Exp
input = do
  T.whiteSpace lexer
  e <- expression
  eof
  return e

expression = do e <- simpleExpression
                e2 <- option e (do
                                 x <- many1 (
                                             do
                                               token ";"
                                               y <- option (ESym "Null") simpleExpression
                                               return y
                                            )
                                 return $ EFunc (ESym "Compound") (e:x)
                               )
                return e2


listExpression = do
  x <- between (token "{") (token "}") (expression `sepBy` (token ","))
  return $ list x

functionCall = do
  e <- primaryExpression
  arglists <- many $ between (token "[") (token "]") argumentList
  return $ chain e arglists where
               chain e [] = e
               chain e (x:[]) = EFunc e x
               chain e (x:xs) = chain (EFunc e x) xs

argumentList = expression `sepBy` (token ",")

blank = do token "_"
           x <- option [] $ (\y -> [y]) <$>  symbol
           return $ EFunc (ESym "Blank") x

blankSequence = do token "__"
                   x <- option [] $ (\y -> [y]) <$>  symbol
                   return $ EFunc (ESym "Repeated") [EFunc (ESym "Blank") x]

blankNullSequence = do token "___"
                       x <- option [] $ (\y -> [y]) <$>  symbol
                       return $ EFunc (ESym "RepeatedNull") [EFunc (ESym "Blank") x]


-- | Parses a string and errors if it fails to parse correctly.
parse' s = case (parse input "(unknown)" s) of
             Left err -> error $ show err
             Right e -> e
