-- Copyright (c) 2010, Adam Crume
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
--     * Redistributions of source code must retain the above copyright notice,
--       this list of conditions and the following disclaimer.
--     * Redistributions in binary form must reproduce the above copyright
--       notice, this list of conditions and the following disclaimer in the
--       documentation and/or other materials provided with the distribution.
--     * Neither the name of the University of California nor the names of its
--       contributors may be used to endorse or promote products derived from
--       this software without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.


module Ein.Data where


import Control.Monad.State
import qualified Data.Map as Map
import qualified Data.Set as Set


data Exp = EInt Integer
         | EFunc Exp [Exp]
         | ESym String
         | EStr String
         deriving (Eq)

instance Show Exp where
    show x = eshow x 0


instance Num Exp where
    (+) x y = EFunc (ESym "Plus") [x, y]
    (*) x y = EFunc (ESym "Times") [x, y]
    fromInteger x = EInt x

-- Necessary so we can store Exp values in maps and sets
instance Ord Exp where
    compare (EInt _) (EFunc _ _) = LT
    compare (EInt _) (ESym _) = LT
    compare (EInt _) (EStr _) = LT
    compare (EStr _) (ESym _) = LT
    compare (EStr _) (EFunc _ _) = LT
    compare (ESym _) (EFunc _ _) = LT
    compare (EInt x) (EInt y) = compare x y
    compare (EStr x) (EStr y) = compare x y
    compare (ESym x) (ESym y) = compare x y
    compare (EFunc f1 xs1) (EFunc f2 xs2) = case (compare f1 f2) of
                                              LT -> LT
                                              GT -> GT
                                              EQ -> compare xs1 xs2
    compare x y = case (compare y x) of
                    LT -> GT
                    EQ -> EQ
                    GT -> LT


-- | Stores the state for evaluation.
data Context = Context {
      contextAtts :: Map.Map Exp (Set.Set Exp), -- ^ Maps base symbols to attributes (usually symbols)
      contextValues :: Map.Map Exp [Rule],  -- ^ Maps base symbols to rules used to evalute expressions whose base symbol is the map key
      contextEagerRules :: [Rule], -- ^ Rules which are tried for every expression and are tried before the value rules are tried
      contextLazyRules :: [Rule], -- ^ Rules which are tried for every expression and are tried after the value rules are tried
      contextDepth :: Int, -- ^ Evaluation depth, equal to the length of contextEvals
      contextEvals :: [Exp] -- ^ Stack of expressions currently being evaluated
    }


-- | Default context with no rules or attributes defined.
emptyContext :: Context
emptyContext = Context Map.empty Map.empty [] [] 0 []


type EC a = State Context a

type ECIO a = StateT Context IO a


-- | A rule function returns Nothing if it does not apply to the given expression.  Otherwise, it returns Just _ with the resulting expression and the (possibly) modified context.
type RuleFunction = Exp -> Context -> Maybe (Exp, Context)

data Rule = Rule {
      ruleString :: String,
      ruleFunc :: RuleFunction
    }

instance Show Rule where
    show r = ruleString r


-- | Converts an expression to a string.
eshow :: Exp -- ^ Expression to convert
      -> Integer -- ^ Precedence of the surrounding expression.  If it is too high, the expression will be surrounded with parentheses.
      -> String -- ^ String value of the expression
eshow (EInt e) i = if e < 0 then '(' : show e ++ ")" else show e
eshow (ESym x) i = x
eshow (EStr x) i = "\"" ++ x ++ "\"" -- TODO: Escape special characters
eshow e@(EFunc (ESym "Times") (x:xs)) i | i <= 3    = eshow x 4 ++ concatMap (\x -> " * " ++ eshow x 4) xs
                                        | otherwise = "(" ++ eshow e 0 ++ ")"
eshow e@(EFunc (ESym "Plus") (x:xs)) i | i <= 2    = eshow x 3 ++ concatMap (\x -> " + " ++ eshow x 3) xs
                                       | otherwise = "(" ++ eshow e 0 ++ ")"
eshow e@(EFunc (ESym "Equal") (x:y:[])) i | i <= 1    = eshow x 2 ++ " == " ++ eshow y 2
                                          | otherwise = "(" ++ eshow e 0 ++ ")"
eshow e@(EFunc f []) i = eshow f 10 ++ "[]"
eshow e@(EFunc f (x:[])) i = eshow f 10 ++ "[" ++ eshow x 0 ++ "]"
eshow e@(EFunc f (x:xs)) i = eshow f 10 ++ "[" ++ eshow x 0 ++ concatMap (\y -> ", " ++ eshow y 0) xs ++ "]"


-- | Converts an expression to a string using only function applications (no special rules for Plus, Times, etc.).
showExpanded :: Exp -> String
showExpanded (EInt e) = show e
showExpanded (ESym e) = e
showExpanded (EStr e) = "\"" ++ e ++ "\"" -- TODO: Escape special characters
showExpanded (EFunc f []) = showExpanded f ++ "[]"
showExpanded (EFunc f (x:[])) = showExpanded f ++ "[" ++ showExpanded x ++ "]"
showExpanded (EFunc f (x:xs)) = showExpanded f ++ "[" ++ showExpanded x ++ concatMap (\y -> ", " ++ showExpanded y) xs ++ "]"


-- | Returns the head of an expression.
expHead :: Exp -> Exp
expHead (EFunc head _) = head
expHead (EInt _) = (ESym "Integer")
expHead (ESym _) = (ESym "Symbol")
expHead (EStr _) = (ESym "String")


-- | Returns a single expression as-is, or multiple expressions wrapped in a Sequence.
squeeze :: [Exp] -> Exp
squeeze (x:[]) = x
squeeze x = EFunc (ESym "Sequence") x


-- | Adds attributes to an expression (normally a symbol).
addAtts :: Exp -- ^ Expression to attach the attributes to
        -> [Exp] -- ^ Attributes to attach
        -> Context -- ^ Context to modify
        -> Context -- ^ Modified context
addAtts e as c =
    let
        s = Map.lookup e (contextAtts c)
        s' = case s of
               Nothing -> Set.fromList as
               Just x -> Set.union x (Set.fromList as)
        m' = Map.insert e s' (contextAtts c)
    in
      c {contextAtts = m'}


-- | Adds an attribute to an expression (normally a symbol).
addAtt :: Exp -- ^ Expression to attach the attribute to
       -> Exp -- ^ Attribute to attach
       -> Context -- ^ Context to modify
       -> Context -- ^ Modified context
addAtt e a c =
    let
        s = Map.lookup e (contextAtts c)
        s' = case s of
               Nothing -> Set.singleton a
               Just x -> Set.insert a x
        m' = Map.insert e s' (contextAtts c)
    in
      c {contextAtts = m'}


-- | Note that values are added to the end of the list, not the beginning.  This changes rule addition time from O(1) to O(n), but it makes eval faster.
--   In any case, eval needs to apply rules in FIFO order: first defined, first applied.
addVal :: Exp -> Rule -> Context -> Context
addVal e r c =
    let
        rs = Map.lookup e (contextValues c)
        rs' = case rs of
               Nothing -> [r]
               Just x -> x++[r]
        m' = Map.insert e rs' (contextValues c)
    in
      c {contextValues = m'}


-- | Pushes an expression on the eval stack.  Used for debugging runaway evals.
contextPush :: Exp -> EC ()
contextPush e =
    do
      c <- get
      d <- return $ (contextDepth c) + 1
      evals <- return $ e:(contextEvals c)
      put $ if d<100
            then c {contextDepth = d, contextEvals = evals}
            else error ("Too many evals: " ++ concatMap (\x->(show x) ++ "\n") evals)


-- | Pops an expression off the eval stack.
contextPop :: EC ()
contextPop =
    do
      c <- get
      d <- return $ (contextDepth c) - 1
      (_:evals) <- return $ if d>=0
                            then contextEvals c
                            else error "Too many contextPop calls"
      put $ c {contextDepth = d, contextEvals = evals}


-- | Creates a List expression.
list :: [Exp] -> Exp
list es = EFunc (ESym "List") es
