-- Copyright (c) 2010, Adam Crume
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
--     * Redistributions of source code must retain the above copyright notice,
--       this list of conditions and the following disclaimer.
--     * Redistributions in binary form must reproduce the above copyright
--       notice, this list of conditions and the following disclaimer in the
--       documentation and/or other materials provided with the distribution.
--     * Neither the name of the University of California nor the names of its
--       contributors may be used to endorse or promote products derived from
--       this software without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.


module DepAnalysis (
                    collectDependencies
                   ) where

import Bag
import Data.List
import Data.Map (Map)
import qualified Data.Map as Map
import Data.Set (Set)
import qualified Data.Set as Set
import GHC
import Outputable
import Var


-- | Converts an Outputable to a string.  This should be in the Outputable module, but maybe I'm using an old version:
showPpr :: Outputable a => a -> String
showPpr = showSDoc . ppr


-- | Takes a binding and a map of function names to dependencies and returns the updated map.
doBind :: Located (HsBind Var) -> Map String (Set String) -> Map String (Set String)
doBind (L _ (AbsBinds tvs dicts exports binds)) deps = foldrBag doBind deps binds
doBind (L _ (FunBind fid finfix matches cofn fvs tick)) deps =
    let
        key = showPpr fid
        set = doMatchGroup matches
        set' = case Map.lookup key deps of
                 Nothing -> set
                 Just s -> Set.union s set
    in
      Map.insert key set' deps
doBind bind deps = deps


-- | Takes a match group and returns free variables.
doMatchGroup :: OutputableBndr id => MatchGroup id -> Set String
doMatchGroup (MatchGroup matches type_) = foldr (\m s -> Set.union s (doMatch m)) Set.empty matches


-- | Takes a match and returns free variables.
doMatch :: OutputableBndr id => Located (Match id) -> Set String
doMatch (L _ (Match pat mtype grhs)) =
    let
        set = doGRHSs grhs
        argSet = foldr (\p s -> Set.union s (doPat p)) Set.empty pat
    in
      Set.difference set argSet


-- | Takes a GRHSs and returns free variables.
doGRHSs :: OutputableBndr id => GRHSs id -> Set String
doGRHSs (GRHSs rhss binds) = doLGRHS (rhss !! 0) -- We should handle cases with multiple guards


-- | Takes an LGRHS and returns free variables.
doLGRHS :: OutputableBndr id => LGRHS id -> Set String
doLGRHS (L _ (GRHS stmts e)) = doLHSExpr e -- We should handle guards


-- | Takes an expression and returns free variables.
doLHSExpr :: OutputableBndr id => LHsExpr id -> Set String
doLHSExpr (L _ (HsVar v)) = Set.singleton $ showPpr v
doLHSExpr (L _ (HsOverLit v)) = Set.empty
doLHSExpr (L _ (HsApp f x)) = Set.union (doLHSExpr f) (doLHSExpr x)
doLHSExpr (L _ (OpApp x op fy y)) = Set.union (doLHSExpr x) (doLHSExpr y)
doLHSExpr (L _ (HsPar x)) = doLHSExpr x
doLHSExpr (L _ (HsCase e matchGroup)) = Set.union (doLHSExpr e) (doMatchGroup matchGroup)
doLHSExpr (L _ x) = error $ "No doLHSExpr for " ++ (showSDocDebug $ ppr x)


-- | Takes a pattern and returns variables bound in the pattern.
doPat :: OutputableBndr id => LPat id -> Set String
doPat (L _ (NPat n _ _)) = Set.empty
doPat (L _ (VarPat v)) = Set.singleton $ showPpr v
doPat (L _ (ParPat p)) = doPat p
doPat (L _ (ConPatOut con tvs dicts binds args ty)) =
            foldr (\a s -> Set.union s (doPat a)) Set.empty $ hsConPatArgs args
doPat x = error $ "doPat not yet supported for " ++ showPpr x


-- | Removes a specific namespace from a string if it is present.
trimNS ns s =
    if (ns ++ ".") `isPrefixOf` s
    then drop (length (ns ++ ".")) s
    else s


-- | Finds the transitive closure of dependencies for a function.  In other words, if f depends on g which depends on h, the result will have f depend on h.
depsTC :: String -- ^ Module name
       -> Map String (Set String) -- ^ Dependency map: function names to set of names of dependencies
       -> String -- ^ Function to return dependencies for
       -> Set String -- ^ Dependencies collected so far.  Top level call should pass in Set.empty.
       -> Set String -- ^ Dependencies of the function.
depsTC modName deps f collected =
    if Set.member f collected
    then collected
    else
        let
            set = case Map.lookup f deps of
                    Just s -> s
                    Nothing -> error $ "Function was not scanned for dependencies: " ++ f
        in
            Set.fold (depsTC modName deps) (Set.insert f collected) set


-- | Calculates the dependencies of a function, in other words, all functions that may get called when it is called.
collectDependencies :: String -- ^ Module name
                    -> [Located (HsBind Var)] -- ^ Bindings
                    -> String -- ^ Name of the function
                    -> Set String -- ^ Functions it depends on
collectDependencies modName binds f =
    let
        deps = foldr doBind Map.empty binds
        deps' = Map.map (Set.map (trimNS modName)) deps
    in
      depsTC modName deps' f Set.empty
